import numpy as np
from matplotlib import pyplot as plt
import matplotlib
from matplotlib.patches import Polygon
from matplotlib.collections import PatchCollection
import matplotlib.gridspec as gridspec


############################################################################################################################
#   Code for producing Figure 3 in the main text.
#   The following data is used in this figure:
#   -- theoryFreediffxz.txt, x-z projection of the diffusion contour for the free diffusion theory with \mu_c = 0 k_B T
#   -- theoryConstdiffxz_3kbT.txt, x-z projection of the diffusion contour for the constrained diffusion theory with \mu_c = 3 k_B T
#   -- theoryConstdiffxz.txt, x-z projection of the diffusion contour for the constrained diffusion theory with \mu_c = 5 k_B T
#   -- theoryConstdiffxz_12kbT.txt, x-z projection of the diffusion contour for the constrained diffusion theory with \mu_c = 12 k_B T
#
#   -- cont_xz_0kbT.txt, x-z projection of the diffusion contour measured in BD simulations with \mu_c = 0 k_B T
#   -- cont_xz_3kbT.txt, x-z projection of the diffusion contour measured in BD simulations with \mu_c = 3 k_B T
#   -- cont_xz_5kbT.txt, x-z projection of the diffusion contour measured in BD simulations with \mu_c = 5 k_B T
#   -- cont_xz_12kbT.txt, x-z projection of the diffusion contour measured in BD simulations with \mu_c = 12 k_B T
#
#   -- andrecka_experimental_data.csv, diffusion contour measured experimentally by Andrecka et al. (2015)
#
############################################################################################################################



levels = np.linspace(-0.00000000001*10**(-4),6.75*10**(-4),14)
axisLabelSize = 18
pltLabelSize=15

params = {'backend': 'ps'}

plt.rcParams.update(params)


fig = plt.figure(figsize=(20, 7.5))




outer = gridspec.GridSpec(9, 11, wspace=1.,hspace=1.)

theory = gridspec.GridSpecFromSubplotSpec(1, 4,
                    subplot_spec=outer[:4,1:9], wspace=0.3)

cbar = gridspec.GridSpecFromSubplotSpec(1, 1,
                    subplot_spec=outer[1:7,0])


simulation = gridspec.GridSpecFromSubplotSpec(1, 4,
                    subplot_spec=outer[5:,1:9], wspace=0.3)

experiment = gridspec.GridSpecFromSubplotSpec(1, 1,
                    subplot_spec=outer[2:6,9:])

ax = plt.Subplot(fig, theory[0, 0])
fig.add_subplot(ax)

z, x, probxz = np.transpose(np.loadtxt('theory_diffusion_contours/theoryFreediffxz.txt'))

numz = np.count_nonzero(z==z[1])
numx = np.count_nonzero(x==x[1])
z = z[::numz]
x = x[:int(len(x)/numx)]
probxz = np.transpose(np.reshape(probxz,[numx, numz]))


ax.contourf(z, x, probxz, levels, cmap=plt.cm.YlOrBr)
ax.contour(z, x, probxz, levels, colors='k')
ax.set_xlabel(r'$z$', size=axisLabelSize,labelpad=0)
ax.set_ylabel(r'$x$', size=axisLabelSize,labelpad=-5)
ax.text(0.99,0.99, r'$\mathcal{P}(x,z)$', size=pltLabelSize, transform=ax.transAxes, horizontalalignment='right', verticalalignment='top')
ax.text(0.01, 0.01, r'$\mu_c = 0$', size=pltLabelSize, transform=ax.transAxes, horizontalalignment='left', verticalalignment='bottom', color='grey')
ax.locator_params(nbins=5)
ax.text(0.01, 0.99, 'A', size=pltLabelSize, transform=ax.transAxes,
              horizontalalignment='left', verticalalignment='top')


ax = plt.Subplot(fig, theory[0, 1])
fig.add_subplot(ax)
ax.set_title(r'Analytical Theory', size=26, y=1.05,x=1.05)
z, x, probxz = np.transpose(np.loadtxt('theory_diffusion_contours/theoryConstdiffxz_3kbT.txt'))

numz = np.count_nonzero(z==z[1])
numx = np.count_nonzero(x==x[1])
z = z[::numz]
x = x[:int(len(x)/numx)]
probxz = np.transpose(np.reshape(probxz,[numx, numz]))
probxz = probxz

colorBarAxs = ax.contourf(z,x, probxz, levels, cmap=plt.cm.YlOrBr)
ax.contour(z,x, probxz, levels, colors='k')
ax.set_xlabel(r'$z$', size=axisLabelSize,labelpad=0)
ax.set_ylabel(r'$x$', size=axisLabelSize,labelpad=-5)
ax.text(0.99,0.99, r'$\mathcal{P}(x,z)$', size=pltLabelSize, transform=ax.transAxes, horizontalalignment='right', verticalalignment='top')
ax.text(0.01, 0.01, r'$\mu_c = 3$', size=pltLabelSize, transform=ax.transAxes, horizontalalignment='left', verticalalignment='bottom', color='grey')
ax.locator_params(nbins=5)
ax.text(0.01, 0.99, 'B', size=pltLabelSize, transform=ax.transAxes,
              horizontalalignment='left', verticalalignment='top')

ax = plt.Subplot(fig, theory[0, 2])
fig.add_subplot(ax)

z, x, probxz = np.transpose(np.loadtxt('theory_diffusion_contours/theoryConstdiffxz.txt'))

numz = np.count_nonzero(z==z[1])
numx = np.count_nonzero(x==x[1])
z = z[::numz]
x = x[:int(len(x)/numx)]
probxz = np.transpose(np.reshape(probxz,[numx, numz]))


ax.contourf(z,x, probxz, levels, cmap=plt.cm.YlOrBr)
ax.contour(z,x, probxz, levels, colors='k')
ax.set_xlabel(r'$z$', size=axisLabelSize,labelpad=0)
ax.set_ylabel(r'$x$', size=axisLabelSize,labelpad=-5)
ax.text(0.99,0.99, r'$\mathcal{P}(x,z)$', size=pltLabelSize, transform=ax.transAxes, horizontalalignment='right', verticalalignment='top')
ax.text(0.01, 0.01, r'$\mu_c = 5$', size=pltLabelSize, transform=ax.transAxes, horizontalalignment='left', verticalalignment='bottom')
ax.locator_params(nbins=5)
ax.text(0.01, 0.99, 'C', size=pltLabelSize, transform=ax.transAxes,
              horizontalalignment='left', verticalalignment='top')
# ax.clabel(CS, inline=1, fontsize=10, fmt='%1.9f')

ax = plt.Subplot(fig, theory[0, 3])
fig.add_subplot(ax)


z, x, probxz = np.transpose(np.loadtxt('theory_diffusion_contours/theoryConstdiffxz_12kbT.txt'))

numz = np.count_nonzero(z==z[1])
numx = np.count_nonzero(x==x[1])
z = z[::numz]
x = x[:int(len(x)/numx)]
probxz = np.transpose(np.reshape(probxz,[numx, numz]))


ax.contourf(z,x, probxz, levels, cmap=plt.cm.YlOrBr)
ax.contour(z,x, probxz, levels, colors='k')
ax.set_xlabel(r'$z$', size=axisLabelSize,labelpad=0)
ax.set_ylabel(r'$x$', size=axisLabelSize,labelpad=-5)
ax.text(0.99,0.99, r'$\mathcal{P}(x,z)$', size=pltLabelSize, transform=ax.transAxes, horizontalalignment='right', verticalalignment='top')
ax.text(0.01, 0.01, r'$\mu_c = 12$', size=pltLabelSize, transform=ax.transAxes, horizontalalignment='left', verticalalignment='bottom', color='grey')
ax.locator_params(nbins=5)
ax.text(0.01, 0.99, 'D', size=pltLabelSize, transform=ax.transAxes,
              horizontalalignment='left', verticalalignment='top')



ax = plt.Subplot(fig, simulation[0, 0])
fig.add_subplot(ax)

z, x, probxz = np.transpose(np.loadtxt('BD_diffusion_contours/cont_xz_0kbT.txt'))

numz = np.count_nonzero(z==z[1])
numx = np.count_nonzero(x==x[1])
z = z[:int(len(z)/numz)]
x = x[::numx]
probxz = np.reshape(probxz,[numz, numx])

ax.contourf(z, x, probxz, levels, cmap=plt.cm.YlOrBr)
ax.contour(z, x, probxz, levels, colors='k')
ax.set_xlabel(r'$z$', size=axisLabelSize,labelpad=0)
ax.set_ylabel(r'$x$', size=axisLabelSize,labelpad=-3)
ax.text(0.99, 0.99, r'$\mathcal{P}(x,z)$', size=pltLabelSize, transform=ax.transAxes, horizontalalignment='right', verticalalignment='top')
ax.locator_params(nbins=5)
ax.text(0.01, 0.99, 'E', size=pltLabelSize, transform=ax.transAxes,
              horizontalalignment='left', verticalalignment='top')


ax = plt.Subplot(fig, simulation[0, 1])
fig.add_subplot(ax)
ax.set_title(r'Brownian Dynamics Simulations', size=26, y=1.05, x=1.1)


z, x, probxz = np.transpose(np.loadtxt('BD_diffusion_contours/cont_xz_3kbT.txt'))

numz = np.count_nonzero(z==z[1])
numx = np.count_nonzero(x==x[1])
z = z[:int(len(z)/numz)]
x = x[::numx]
probxz = np.reshape(probxz,[numz, numx])

ax.contourf(z, x, probxz, levels, cmap=plt.cm.YlOrBr)
ax.contour(z, x, probxz, levels, colors='k')
ax.set_xlabel(r'$z$', size=axisLabelSize,labelpad=0)
ax.set_ylabel(r'$x$', size=axisLabelSize,labelpad=-3)
ax.text(0.99, 0.99, r'$\mathcal{P}(x,z)$', size=pltLabelSize, transform=ax.transAxes, horizontalalignment='right', verticalalignment='top')
ax.locator_params(nbins=5)
ax.text(0.01, 0.99, 'F', size=pltLabelSize, transform=ax.transAxes,
              horizontalalignment='left', verticalalignment='top')

ax = plt.Subplot(fig, simulation[0, 2])
fig.add_subplot(ax)


z, x, probxz = np.transpose(np.loadtxt('BD_diffusion_contours/cont_xz_5kbT.txt'))

numz = np.count_nonzero(z==z[1])
numx = np.count_nonzero(x==x[1])
z = z[:int(len(z)/numz)]
x = x[::numx]
probxz = np.reshape(probxz,[numz, numx])

ax.contourf(z, x, probxz, levels, cmap=plt.cm.YlOrBr)
ax.contour(z, x, probxz, levels, colors='k')
ax.set_xlabel(r'$z$', size=axisLabelSize,labelpad=0)
ax.set_ylabel(r'$x$', size=axisLabelSize,labelpad=-3)
ax.text(0.99, 0.99, r'$\mathcal{P}(x,z)$', size=pltLabelSize, transform=ax.transAxes, horizontalalignment='right', verticalalignment='top')
ax.locator_params(nbins=5)
ax.text(0.01, 0.99, 'G', size=pltLabelSize, transform=ax.transAxes,
              horizontalalignment='left', verticalalignment='top')


ax = fig.add_subplot(simulation[0, 3])


z, x, probxz = np.transpose(np.loadtxt('BD_diffusion_contours/cont_xz_12kbT.txt'))

numz = np.count_nonzero(z==z[1])
numx = np.count_nonzero(x==x[1])
z = z[:int(len(z)/numz)]
x = x[::numx]
probxz = np.reshape(probxz,[numz, numx])

ax.contourf(z, x, probxz, levels, cmap=plt.cm.YlOrBr)
ax.contour(z, x, probxz, levels, colors='k')
ax.set_xlabel(r'$z$', size=axisLabelSize,labelpad=0)
ax.set_ylabel(r'$x$', size=axisLabelSize,labelpad=-3)
ax.text(0.99, 0.99, r'$\mathcal{P}(x,z)$', size=pltLabelSize, transform=ax.transAxes, horizontalalignment='right', verticalalignment='top')
ax.locator_params(nbins=5)
ax.text(0.01, 0.99, 'H', size=pltLabelSize, transform=ax.transAxes,
              horizontalalignment='left', verticalalignment='top')


ax = fig.add_subplot(experiment[0,0])
ax.set_title(r'Experiment', size=28, y=1.08)
ax.text(0.5,1.01, r'(Andrecka $et \, \, al.$, 2015)', size=13.5, transform=ax.transAxes, horizontalalignment='center', verticalalignment='bottom')
ax.set_xlabel(r'$z$', size=axisLabelSize,labelpad=0)
ax.set_ylabel(r'$x$', size=axisLabelSize,labelpad=-10)
ax.text(0.99,0.99, r'$\mathcal{P}(x, z)$', size=pltLabelSize, transform=ax.transAxes, horizontalalignment='right', verticalalignment='top')
ax.text(0.01, 0.99, 'I', size=pltLabelSize, transform=ax.transAxes,
              horizontalalignment='left', verticalalignment='top')

expData = np.loadtxt(open('experimental_diffusion_contours/andrecka_experimental_data.csv', 'rb'), delimiter=',', skiprows=1)

patches = []
patches2 = []

lims = [[0, 835], [836, 1393], [1396, 1576], [1577, 1638], [1639, 1668], [1669, 1701], [1702, 1901], [1902, 2038], [2039, 2098], [2099, 2123], [2124, len(expData)]]


polygon = Polygon(np.array([[-100,-100],[-100,100],[100,100],[100,-100]]), edgecolor='k', color=None, fill=False)
patches.append(polygon)
patches2.append(polygon)

for start, end in lims:
    polygon = Polygon(expData[start:end], True)
    patches.append(polygon)
    ax.plot(np.append(expData[start:end,0],expData[start,0]), np.append(expData[start:end,1],expData[start,1]) ,'k')


p = PatchCollection(patches, cmap=plt.cm.YlOrBr, alpha=1.)


totalWeight = 1522.54
maxProb = 6.75*10**(-4)

p.set_array(np.array([0, 1/6, 2/6, 3/6, 4/6, 3/6, 3/6, 3/6, 4/6, 5/6, 6/6, 3/6])/totalWeight/maxProb)
p.set_clim([0, 1])

ax.add_collection(p)
ax.set_xlim([-45,65])
ax.set_ylim([-40,75])
ax.set_xticks([-40, -20,0,20,40,60])


ax = fig.add_subplot(cbar[:,:])

norm= matplotlib.colors.Normalize(0, 6.75)
sm = plt.cm.ScalarMappable(norm=norm, cmap=plt.cm.YlOrBr)
sm.set_array([])

cbar = fig.colorbar(sm, ticks=range(7), cax=ax, label=r'Probability ($\times 10^{-4}$)')
cbar.ax.tick_params(labelsize=18)
cbar.ax.set_yticklabels(range(7))
cbar.ax.yaxis.set_ticks_position('left')
cbar.ax.yaxis.set_label_position('left')
cbar.ax.yaxis.label.set_size(20)


plt.savefig('diffusion_figure.pdf',bbox_inches='tight')

plt.show()
